/*
 * Copyright (c) 2016, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */ 

/**
 *\file : nandflash.h
 *
 *\brief: Defines the module register defines for NAND Flash
 *
 * This header file contains the Macros definitions and NAND flash
 *
 * \author Antriksh
 *
 * \version 01a,09aug, An 2016 Created.
 * \version 02a,12Aug,An, 2016 comments and codeupdated for the 
 *                             processor DM8127.
 */

#ifndef _NAND_FLASH_H_
#define _nandflash_H_
/*========================
 * Includes
 *========================
 */
 
#include "DM8127_types.h"
#include "gpio.h"
#include "DM8127_types.h"
/*=========================
 * Defines
 *=========================
 */
//#define DEBUG
/** GPIO register  */
#define GPIO1_BASE_ADD			(0x4804C000)
/* bit selection*/
#define BIT_SEL_GPIO (10u)

#if 0
#define GPIO1_SYSCONFIG		*(UINT32 *) (GPIO1_BASE_ADD + 0x010)
#define GPIO1_CTRL			*(UINT32 *) (GPIO1_BASE_ADD + 0x130)
#define GPIO1_OE			*(UINT32 *) (GPIO1_BASE_ADD + 0x134)
#define GPIO1_DATAIN		*(UINT32 *) (GPIO1_BASE_ADD + 0x138)
#define GPIO1_DATAOUT		*(UINT32 *) (GPIO1_BASE_ADD + 0x13C)
#define GPIO1_CLEARDATAOUT	*(UINT32 *) (GPIO1_BASE_ADD + 0x190)
#define GPIO1_SETDATAOUT	*(UINT32 *) (GPIO1_BASE_ADD + 0x194)
#endif
/** Base address of the NAND flash */
#define NAND_FLASH_BASE          (0x50000000)  // Low CS

/** Number of available Using blocks */
#define NAND_FLASH_BLOCKS        (2048u)

/** Number of pages in a block */
#define NAND_FLASH_PAGESPERBLCK  (64u)

/** Total number os pages available in the NAND Flash */
#define NAND_FLASH_PAGES         (NAND_FLASH_BLOCKS * NAND_FLASH_PAGESPERBLCK)

/** Size of the pages os a block */
#define NAND_FLASH_PAGESIZE      (2048u)

/** spare area avilale into page */
#define NAND_FLASH_SPARESIZE     (64u)

/** Nand flash page size with spare area */
#define NAND_FLASH_PAGE_SIZE_WITH_SPARE_AREA  \
                        (NAND_FLASH_PAGESIZE + NAND_FLASH_SPARESIZE)

/**
 *  This count can change according to the User. And it is depended
 *  on the how many numbers of block that user wants to test at a
 *  time.
 */

#define NAND_FLASH_BLOCKS_TEST_CNT  (4u)//NAND_FLASH_BLOCKS  //

/** Command latch enable address */
#define NANDFLASH_CLE   *( volatile UINT16* )( NAND_FLASH_BASE + 0x7C )

/** Address latch enable */
#define NANDFLASH_ALE   *( volatile UINT16* )( NAND_FLASH_BASE + 0x80 )

/** Flash Data access address  */
#define NANDFLASH_DATA  *( volatile UINT16* )( NAND_FLASH_BASE + 0x84 )

////////////////////////////////
/// Command for the NAND flash
///////////////////////////////

/** Nand flash read ID command */
#define CMD_READID                  (0x0090)

/** command cycle#1 for the read operaion */
#define CMD_READ                    (0x0000)
/** command cycle#2 for the read operaion */
#define CMD_READ_CONFIRM            (0x0030)

/** command cycle#1 for the write/program  operation */
#define CMD_PROGRAM                 (0x0080)
/** command cycle#2 for the write/program  operation */
#define CMD_PROGRAM_CONFIRM         (0x0010)

/** command cycle#1 for the Erase operaion */
#define CMD_ERASE                   (0x0060)
/** command cycle#2 for the Erase operaion */
#define CMD_ERASE_CONFIRM           (0x00D0)

/** command done statsu read  */
#define CMD_STATUS                  (0x0070)

/** command for reset the flash */
#define CMD_RESET                   (0x00ff)

/*
 *======================
 *Function Declarations
 *======================
 */

/***
 * \brief: NAND Flash #wp test function.
 *
 * \b \Descr: This will send the command to the NAND flash to check the status
 * \n         for the NAND flash write protect or not.
 *
 * \param: VOID
 * \return:VOID
 */


void nand_isWriteProtect
(
	void
);

 /***
  * \brief: NAND Flash initialization function.
  *
  * \b \descr: it initialize the NAND Flash chip to communicae with
  * \n         the user test.
  *
  * \param: VOID
  * \return: SUCCESS for successful initialize
  *          FAILED  Failed to initialize
  */


STATUS nand_init
(
		void
);

/***
 * \brief: Routine to Erase the Nand Flash
 *
 * \b \Descr: This routine Erases a block of Nand Flash .This function takes Block
 * \n         num as input parameter.Reads erase status and checks for successfull
 * \n         erase operation.
 *
 * \param:   blocknum 		[IN]		Block to be erased
 *
 * \return:  SUCCESS for success
 * \n        FAILED for error
 *
 */

STATUS nand_eraseBlock
(
	UINT32 blocknum
);

/***
 * \brief: Routine to read from  Nand Flash
 *
 * \b \Descr: This routine Reads a page from Nand Flash .This function takes Page
 * \n         number as input parameter along with read buffer pointer and length
 * \n         of buffer.
 *
 *
 * \param:  pagenum 	  [IN]		Pagenumber to read from
 * \n       *buffer       [IN]      Buffer pointer to copy read data
 * \n       len           [IN]      Length of the buffer
 *
 *
 * \return: SUCCESS for success
 * \n       FAILED for error
 */
STATUS nand_readPage
(
	UINT32 pagenum,
	UINT16 *buf,
	UINT16 len
);

/***
 * \brief: Routine to Write into  Nand Flash
 *
 * \b \Descr: This routine Writes a page to  Nand Flash. This function takes Page
 * \n         number as input parameter along with  buffer pointer and length of
 * \n         buffer.
 *
 *
 * \param:   pagenum 	    [IN]       Page to be written
 * \n        *buffer        [IN]       Buffer pointer which contains data to be written
 * \n        len            [IN]       Length of the buffer
 *
 *
 * \return:   SUCCESS for success
 * \n         FAILED for error
 */

STATUS nand_writePage
(
	UINT32 pagenum,
	UINT16 *buf,
	UINT16 len
);

/**
 *\brief : Routine to Test Nandflash device .
 *
 *         This routine tests first 4 blocks of device without Spare area.
 *         Uncommenting the  loop tests with spare area.This routine first
 *         erases the blocks.Writes into the device and read the same and
 *         Compares both.
 *
 *@param   VOID
 *\return :   NAND_TEST_PASS  Successfully pass
 *          NAND_TEST_FAIL  NAND test Failed
 */

STATUS nandflash_test
(
	void
);


#endif /* _nandflash_H_ */
